# database.py - Updated with branch-based licensing
import sqlite3
from datetime import datetime, timedelta
import hashlib
import uuid

class DatabaseManager:
    def __init__(self, db_name='medical_center.db'):
        self.db_name = db_name
        self.init_database()
    
    def get_connection(self):
        return sqlite3.connect(self.db_name)
    
    def init_database(self):
        conn = self.get_connection()
        cursor = conn.cursor()
        
        # Create patients table
        cursor.execute('''
            CREATE TABLE IF NOT EXISTS patients (
                id INTEGER PRIMARY KEY AUTOINCREMENT,
                mr_number TEXT UNIQUE,
                name TEXT NOT NULL,
                cnic TEXT NOT NULL,
                mobile TEXT NOT NULL,
                dob TEXT NOT NULL,
                age INTEGER,
                gender TEXT NOT NULL,
                address TEXT,
                created_date TEXT
            )
        ''')
        
                # Create hospital settings table
        cursor.execute('''
            CREATE TABLE IF NOT EXISTS hospital_settings (
                id INTEGER PRIMARY KEY AUTOINCREMENT,
                setting_name TEXT UNIQUE NOT NULL,
                setting_value TEXT,
                created_date TEXT,
                updated_date TEXT
            )
        ''')

        # Insert default settings if not exists
        cursor.execute("SELECT COUNT(*) FROM hospital_settings WHERE setting_name = 'hospital_name'")
        if cursor.fetchone()[0] == 0:
            default_settings = [
                ('hospital_name', 'Medical Center Hospital', datetime.now().strftime('%Y-%m-%d %H:%M:%S')),
                ('hospital_address', '123 Main Street, City, Country', datetime.now().strftime('%Y-%m-%d %H:%M:%S')),
                ('hospital_phone', '+1-234-567-8900', datetime.now().strftime('%Y-%m-%d %H:%M:%S')),
                ('hospital_timings', 'Mon-Sat: 9:00 AM - 8:00 PM, Sun: 10:00 AM - 2:00 PM', datetime.now().strftime('%Y-%m-%d %H:%M:%S')),
                ('token_prefix', 'TKT', datetime.now().strftime('%Y-%m-%d %H:%M:%S')),
                ('token_suffix', '', datetime.now().strftime('%Y-%m-%d %H:%M:%S')),
                ('billing_invoice_prefix', 'INV', datetime.now().strftime('%Y-%m-%d %H:%M:%S')),
                ('billing_invoice_suffix', '', datetime.now().strftime('%Y-%m-%d %H:%M:%S')),
                ('billing_footer_text', 'Thank you for choosing our services!', datetime.now().strftime('%Y-%m-%d %H:%M:%S'))
            ]
            
            for setting_name, setting_value, created_date in default_settings:
                cursor.execute('''
                    INSERT INTO hospital_settings (setting_name, setting_value, created_date, updated_date)
                    VALUES (?, ?, ?, ?)
                ''', (setting_name, setting_value, created_date, created_date))
        
        # Create doctors table
        cursor.execute('''
            CREATE TABLE IF NOT EXISTS doctors (
                id INTEGER PRIMARY KEY AUTOINCREMENT,
                doctor_id TEXT UNIQUE,
                name TEXT NOT NULL,
                specialization TEXT,
                mobile TEXT,
                email TEXT,
                qualification TEXT,
                experience INTEGER,
                fee REAL,
                created_date TEXT
            )
        ''')
        
        # Create appointments table
        cursor.execute('''
            CREATE TABLE IF NOT EXISTS appointments (
                id INTEGER PRIMARY KEY AUTOINCREMENT,
                appointment_id TEXT UNIQUE,
                patient_id INTEGER,
                doctor_id INTEGER,
                appointment_date TEXT,
                appointment_time TEXT,
                status TEXT DEFAULT 'Scheduled',
                notes TEXT,
                created_date TEXT,
                FOREIGN KEY (patient_id) REFERENCES patients (id),
                FOREIGN KEY (doctor_id) REFERENCES doctors (id)
            )
        ''')
        
        # Create medicines table
        cursor.execute('''
            CREATE TABLE IF NOT EXISTS medicines (
                id INTEGER PRIMARY KEY AUTOINCREMENT,
                medicine_id TEXT UNIQUE,
                name TEXT NOT NULL,
                generic_name TEXT,
                manufacturer TEXT,
                category TEXT,
                unit_price REAL,
                stock_quantity INTEGER,
                expiry_date TEXT,
                created_date TEXT
            )
        ''')
        
        # Create prescriptions table
        cursor.execute('''
            CREATE TABLE IF NOT EXISTS prescriptions (
                id INTEGER PRIMARY KEY AUTOINCREMENT,
                prescription_id TEXT UNIQUE,
                patient_id INTEGER,
                doctor_id INTEGER,
                appointment_id INTEGER,
                prescription_date TEXT,
                symptoms TEXT,
                diagnosis TEXT,
                medicine TEXT,
                dosage TEXT,
                duration TEXT,
                advice TEXT,
                created_date TEXT,
                FOREIGN KEY (patient_id) REFERENCES patients (id),
                FOREIGN KEY (doctor_id) REFERENCES doctors (id),
                FOREIGN KEY (appointment_id) REFERENCES appointments (id)
            )
        ''')
        
        # Create billing table
        cursor.execute('''
            CREATE TABLE IF NOT EXISTS billing (
                id INTEGER PRIMARY KEY AUTOINCREMENT,
                bill_id TEXT UNIQUE,
                patient_id INTEGER,
                doctor_id INTEGER,
                appointment_id INTEGER,
                prescription_id INTEGER,
                consultation_fee REAL,
                medicine_cost REAL,
                total_amount REAL,
                discount REAL DEFAULT 0,
                tax REAL DEFAULT 0,
                net_amount REAL,
                payment_method TEXT,
                payment_status TEXT DEFAULT 'Pending',
                bill_date TEXT,
                created_date TEXT,
                FOREIGN KEY (patient_id) REFERENCES patients (id),
                FOREIGN KEY (doctor_id) REFERENCES doctors (id),
                FOREIGN KEY (appointment_id) REFERENCES appointments (id),
                FOREIGN KEY (prescription_id) REFERENCES prescriptions (id)
            )
        ''')
        
        # Create bill_items table
        cursor.execute('''
            CREATE TABLE IF NOT EXISTS bill_items (
                id INTEGER PRIMARY KEY AUTOINCREMENT,
                bill_id INTEGER,
                medicine_id INTEGER,
                quantity INTEGER,
                unit_price REAL,
                total_price REAL,
                created_date TEXT,
                FOREIGN KEY (bill_id) REFERENCES billing (id),
                FOREIGN KEY (medicine_id) REFERENCES medicines (id)
            )
        ''')
        
        # Create users table
        cursor.execute('''
            CREATE TABLE IF NOT EXISTS users (
                id INTEGER PRIMARY KEY AUTOINCREMENT,
                username TEXT UNIQUE NOT NULL,
                password TEXT NOT NULL,
                role TEXT NOT NULL,
                is_activated BOOLEAN DEFAULT 0,
                trial_expiry TEXT,
                is_trial BOOLEAN DEFAULT 1,
                registration_key TEXT,
                branch_id INTEGER DEFAULT 1,
                created_date TEXT
            )
        ''')
        
        # Create branches table
        cursor.execute('''
            CREATE TABLE IF NOT EXISTS branches (
                id INTEGER PRIMARY KEY AUTOINCREMENT,
                branch_name TEXT NOT NULL,
                branch_code TEXT UNIQUE NOT NULL,
                address TEXT,
                contact_number TEXT,
                is_active BOOLEAN DEFAULT 1,
                is_activated BOOLEAN DEFAULT 0,
                trial_expiry TEXT,
                is_trial BOOLEAN DEFAULT 1,
                registration_key TEXT,
                created_date TEXT
            )
        ''')
        
        # Create backup settings table
        cursor.execute('''
            CREATE TABLE IF NOT EXISTS backup_settings (
                id INTEGER PRIMARY KEY AUTOINCREMENT,
                service_type TEXT,
                api_key TEXT,
                backup_path TEXT,
                last_backup TEXT,
                auto_backup BOOLEAN DEFAULT 0
            )
        ''')
        
        # Create PDF settings table
        cursor.execute('''
            CREATE TABLE IF NOT EXISTS pdf_settings (
                id INTEGER PRIMARY KEY AUTOINCREMENT,
                setting_name TEXT UNIQUE,
                background_path TEXT,
                created_date TEXT
            )
        ''')
        
        # Create daily token counter table
        cursor.execute('''
            CREATE TABLE IF NOT EXISTS daily_token_counter (
                id INTEGER PRIMARY KEY AUTOINCREMENT,
                date TEXT UNIQUE,
                last_token_number INTEGER DEFAULT 0
            )
        ''')
        
        # Create temporary registration keys table
        cursor.execute('''
            CREATE TABLE IF NOT EXISTS temp_registration_keys (
                id INTEGER PRIMARY KEY AUTOINCREMENT,
                key TEXT UNIQUE NOT NULL,
                created_date TEXT,
                expiry_date TEXT,
                is_used BOOLEAN DEFAULT 0,
                pc_identifier TEXT
            )
        ''')
        
        conn.commit()
        
        # Create default admin and branch if they don't exist
        self.create_default_branch()
        self.create_default_admin()
        
        conn.close()
    
    def create_default_branch(self):
        """Create default branch if none exists"""
        conn = self.get_connection()
        cursor = conn.cursor()
        
        cursor.execute("SELECT COUNT(*) FROM branches")
        if cursor.fetchone()[0] == 0:
            trial_expiry = (datetime.now() + timedelta(days=30)).strftime('%Y-%m-%d %H:%M:%S')
            cursor.execute('''
                INSERT INTO branches (branch_name, branch_code, address, contact_number, 
                                    is_trial, trial_expiry, created_date)
                VALUES (?, ?, ?, ?, ?, ?, ?)
            ''', ('Main Branch', 'MAIN', 'Hospital Main Address', '123-456-7890', 
                  1, trial_expiry, datetime.now().strftime('%Y-%m-%d %H:%M:%S')))
            conn.commit()
        
        conn.close()
    
    def create_default_admin(self):
        """Create default admin if none exists"""
        conn = self.get_connection()
        cursor = conn.cursor()
        
        cursor.execute("SELECT COUNT(*) FROM users WHERE username = 'admin'")
        if cursor.fetchone()[0] == 0:
            # Get default branch ID
            cursor.execute("SELECT id FROM branches WHERE branch_code = 'MAIN'")
            branch_result = cursor.fetchone()
            branch_id = branch_result[0] if branch_result else 1
            
            trial_expiry = (datetime.now() + timedelta(days=30)).strftime('%Y-%m-%d %H:%M:%S')
            hashed_password = hashlib.sha256("admin123".encode()).hexdigest()
            cursor.execute('''
                INSERT INTO users (username, password, role, branch_id, is_trial, trial_expiry, created_date)
                VALUES (?, ?, ?, ?, ?, ?, ?)
            ''', ("admin", hashed_password, "Administrator", branch_id, 1, trial_expiry, 
                  datetime.now().strftime('%Y-%m-%d %H:%M:%S')))
            conn.commit()
        
        conn.close()
    
    def is_branch_activated(self, branch_id):
        """Check if branch is activated or in valid trial"""
        conn = self.get_connection()
        cursor = conn.cursor()
        
        cursor.execute("SELECT is_activated, trial_expiry, is_trial FROM branches WHERE id = ?", (branch_id,))
        result = cursor.fetchone()
        conn.close()
        
        if not result:
            return False
        
        is_activated, trial_expiry, is_trial = result
        
        if is_activated:
            return True
        
        if is_trial and trial_expiry:
            try:
                trial_end = datetime.strptime(trial_expiry, '%Y-%m-%d %H:%M:%S')
                if datetime.now() < trial_end:
                    return True
            except:
                pass
        
        return False
    
    def get_branch_trial_days(self, branch_id):
        """Get remaining trial days for branch"""
        conn = self.get_connection()
        cursor = conn.cursor()
        
        cursor.execute("SELECT trial_expiry, is_trial FROM branches WHERE id = ?", (branch_id,))
        result = cursor.fetchone()
        conn.close()
        
        if result and result[1]:  # is_trial = 1
            try:
                trial_end = datetime.strptime(result[0], '%Y-%m-%d %H:%M:%S')
                days_left = (trial_end - datetime.now()).days
                return max(0, days_left)
            except:
                pass
        return 0
    
    def activate_branch(self, branch_id, registration_key):
        """Activate branch with registration key"""
        conn = self.get_connection()
        cursor = conn.cursor()
        
        # Check if key is valid and not used
        cursor.execute("SELECT * FROM temp_registration_keys WHERE key = ? AND is_used = 0", (registration_key,))
        key_record = cursor.fetchone()
        
        if key_record:
            # Check if key is not expired
            if key_record[3]:  # expiry_date
                try:
                    expiry_date = datetime.strptime(key_record[3], '%Y-%m-%d %H:%M:%S')
                    if datetime.now() > expiry_date:
                        conn.close()
                        return False, "Registration key has expired"
                except:
                    pass
            
            # Activate branch
            cursor.execute('''
                UPDATE branches 
                SET is_activated = 1, 
                    registration_key = ?, 
                    activation_date = ?, 
                    is_trial = 0
                WHERE id = ?
            ''', (registration_key, datetime.now().strftime('%Y-%m-%d %H:%M:%S'), branch_id))
            
            # Mark key as used
            cursor.execute('''
                UPDATE temp_registration_keys 
                SET is_used = 1, pc_identifier = ?
                WHERE key = ?
            ''', (f"branch_{branch_id}", registration_key))
            
            conn.commit()
            conn.close()
            return True, "Branch activated successfully!"
        
        conn.close()
        return False, "Invalid or already used registration key"
    
    def generate_temp_registration_key(self, days_valid=365):
        """Generate a temporary registration key"""
        conn = self.get_connection()
        cursor = conn.cursor()
        
        timestamp = datetime.now().strftime('%Y%m%d%H%M%S')
        key = f"TEMP-{timestamp}"
        
        expiry_date = (datetime.now() + timedelta(days=days_valid)).strftime('%Y-%m-%d %H:%M:%S')
        
        try:
            cursor.execute('''
                INSERT INTO temp_registration_keys (key, created_date, expiry_date)
                VALUES (?, ?, ?)
            ''', (key, datetime.now().strftime('%Y-%m-%d %H:%M:%S'), expiry_date))
            conn.commit()
            conn.close()
            return key
        except Exception as e:
            conn.close()
            return None
    
    def get_all_temp_keys(self):
        """Get all temporary registration keys"""
        conn = self.get_connection()
        cursor = conn.cursor()
        
        cursor.execute("SELECT * FROM temp_registration_keys ORDER BY created_date DESC")
        result = cursor.fetchall()
        conn.close()
        
        return result
    
    def delete_temp_key(self, key):
        """Delete a temporary registration key"""
        conn = self.get_connection()
        cursor = conn.cursor()
        
        cursor.execute("DELETE FROM temp_registration_keys WHERE key = ?", (key,))
        conn.commit()
        conn.close()
    
    def execute_query(self, query, params=None):
        conn = self.get_connection()
        cursor = conn.cursor()
        if params:
            cursor.execute(query, params)
        else:
            cursor.execute(query)
        result = cursor.fetchall()
        conn.commit()
        conn.close()
        return result
    
    def execute_insert(self, query, params=None):
        conn = self.get_connection()
        cursor = conn.cursor()
        if params:
            cursor.execute(query, params)
        else:
            cursor.execute(query)
        last_id = cursor.lastrowid
        conn.commit()
        conn.close()
        return last_id

# Helper functions
def calculate_age(birth_year):
    try:
        current_year = datetime.now().year
        return current_year - int(birth_year)
    except:
        return 0

def generate_unique_id(prefix):
    timestamp = datetime.now().strftime('%Y%m%d%H%M%S')
    return f"{prefix}{timestamp}"