# app.py - COMPLETELY FIXED VERSION (ALL patient_ ERRORS FIXED)
from datetime import datetime, timedelta
from flask import Flask, render_template, request, redirect, url_for, session, flash, send_file, jsonify
from auth import AuthManager
from database import DatabaseManager
import os
import shutil
from datetime import datetime
from io import BytesIO
import hashlib
from utils.helpers import generate_unique_id, calculate_age
from utils.pdf_generator import PDFGenerator

app = Flask(__name__)
app.secret_key = os.urandom(24)

# Initialize DB and Auth
db_manager = DatabaseManager()
auth_manager = AuthManager(db_manager)

# ============ AUTHENTICATION & LICENSING ROUTES ============

@app.route('/')
def index():
    if 'user' not in session:
        return redirect(url_for('login'))
    return redirect(url_for('dashboard'))

@app.route('/login', methods=['GET', 'POST'])
def login():
    if 'user' in session:
        return redirect(url_for('dashboard'))

    if request.method == 'POST':
        username = request.form.get('username', '').strip()
        password = request.form.get('password', '').strip()

        if not username or not password:
            flash('Please enter both username and password', 'error')
            return render_template('login.html')

        user = auth_manager.authenticate_user(username, password)
        if user:
            branch_id = user[8] if len(user) > 8 else 1
            
            if not db_manager.is_branch_activated(branch_id):
                trial_days = db_manager.get_branch_trial_days(branch_id)
                if trial_days <= 0:
                    # Redirect to activation screen
                    flash('Your branch trial has expired. Please activate with registration key.', 'error')
                    session['pending_activation_branch_id'] = branch_id
                    return redirect(url_for('activate'))
                else:
                    flash(f'Branch trial mode: {trial_days} days remaining', 'warning')

            session['user'] = {
                'id': user[0],
                'username': user[1],
                'role': user[3],
                'branch_id': branch_id,
                'branch_name': user[9] if len(user) > 9 else 'Main Branch',
                'is_branch_activated': db_manager.is_branch_activated(branch_id),
                'trial_days': db_manager.get_branch_trial_days(branch_id)
            }
            return redirect(url_for('dashboard'))
        else:
            flash('Invalid username or password', 'error')

    return render_template('login.html')

@app.route('/logout')
def logout():
    session.pop('user', None)
    if 'pending_activation_branch_id' in session:
        del session['pending_activation_branch_id']
    return redirect(url_for('login'))

@app.route('/activate', methods=['GET', 'POST'])
def activate():
    if 'user' not in session and 'pending_activation_branch_id' not in session:
        return redirect(url_for('login'))

    branch_id = None
    if 'user' in session:
        branch_id = session['user']['branch_id']
    elif 'pending_activation_branch_id' in session:
        branch_id = session['pending_activation_branch_id']

    if not branch_id:
        flash('No branch selected for activation', 'error')
        return redirect(url_for('login'))

    if db_manager.is_branch_activated(branch_id):
        if 'pending_activation_branch_id' in session:
            del session['pending_activation_branch_id']
        return redirect(url_for('dashboard'))

    trial_days = db_manager.get_branch_trial_days(branch_id)

    if request.method == 'POST':
        reg_key = request.form.get('registration_key', '').strip()
        if not reg_key:
            flash('Please enter registration key', 'error')
        else:
            success, message = db_manager.activate_branch(branch_id, reg_key)
            if success:
                if 'user' in session:
                    session['user']['is_branch_activated'] = True
                    flash('Branch activated successfully!', 'success')
                    return redirect(url_for('dashboard'))
                else:
                    del session['pending_activation_branch_id']
                    flash('Branch activated successfully! Please login.', 'success')
                    return redirect(url_for('login'))
            else:
                flash(message, 'error')

    return render_template('activate.html', trial_days=trial_days, branch_id=branch_id)

# ============ DASHBOARD & MANAGEMENT ROUTES ============

@app.route('/dashboard')
def dashboard():
    if 'user' not in session:
        return redirect(url_for('login'))
    user = session['user']
    
    # Check trial/activation
    if not user.get('is_branch_activated') and user.get('trial_days', 0) <= 0:
        flash('Your branch trial has expired. Please activate.', 'error')
        return redirect(url_for('activate'))
    
    return render_template('dashboard.html', user=user)

@app.route('/about')
def about():
    if 'user' not in session:
        return redirect(url_for('login'))
    user = session['user']
    return render_template('about.html', user=user)

@app.route('/backup', methods=['GET', 'POST'])
def backup():
    if 'user' not in session:
        return redirect(url_for('login'))
    
    if request.method == 'POST':
        action = request.form.get('action')
        
        if action == 'local_backup':
            try:
                timestamp = datetime.now().strftime('%Y%m%d_%H%M%S')
                backup_filename = f'medical_center_backup_{timestamp}.db'
                shutil.copy2('medical_center.db', backup_filename)
                return send_file(backup_filename, as_attachment=True)
            except Exception as e:
                flash(f'Backup failed: {str(e)}', 'error')
        
        elif action == 'save_settings':
            service = request.form.get('cloud_service', 'Google Drive')
            api_key = request.form.get('api_key', '')
            backup_path = request.form.get('backup_path', '/medical_center_backups/')
            auto_backup = 'auto_backup' in request.form
            
            try:
                existing = db_manager.execute_query("SELECT COUNT(*) FROM backup_settings")
                if existing[0][0] > 0:
                    db_manager.execute_query('''
                        UPDATE backup_settings 
                        SET service_type = ?, api_key = ?, backup_path = ?, auto_backup = ?
                    ''', (service, api_key, backup_path, auto_backup))
                else:
                    db_manager.execute_insert('''
                        INSERT INTO backup_settings (service_type, api_key, backup_path, auto_backup)
                        VALUES (?, ?, ?, ?)
                    ''', (service, api_key, backup_path, auto_backup))
                flash('Backup settings saved!', 'success')
            except Exception as e:
                flash(f'Failed to save settings: {str(e)}', 'error')
    
    # Load existing settings
    settings = {}
    try:
        result = db_manager.execute_query("SELECT * FROM backup_settings LIMIT 1")
        if result:
            settings = {
                'service_type': result[0][1] or 'Google Drive',
                'api_key': result[0][2] or '',
                'backup_path': result[0][3] or '/medical_center_backups/',
                'auto_backup': bool(result[0][4])
            }
    except:
        pass
    
    return render_template('backup.html', settings=settings)

@app.route('/settings', methods=['GET', 'POST'])
def settings():
    if 'user' not in session:
        return redirect(url_for('login'))
    
    if request.method == 'POST':
        action = request.form.get('action')
        if action == 'set_background':
            if 'background_file' in request.files:
                file = request.files['background_file']
                if file.filename != '':
                    os.makedirs('static/uploads', exist_ok=True)
                    filename = f"prescription_bg_{datetime.now().strftime('%Y%m%d_%H%M%S')}.png"
                    filepath = os.path.join('static/uploads', filename)
                    file.save(filepath)
                    
                    try:
                        existing = db_manager.execute_query("SELECT COUNT(*) FROM pdf_settings WHERE setting_name = 'prescription_background'")
                        if existing[0][0] > 0:
                            db_manager.execute_query("UPDATE pdf_settings SET background_path = ? WHERE setting_name = 'prescription_background'", (filename,))
                        else:
                            db_manager.execute_insert("INSERT INTO pdf_settings (setting_name, background_path, created_date) VALUES (?, ?, ?)", 
                                                    ('prescription_background', filename, datetime.now().strftime('%Y-%m-%d %H:%M:%S')))
                        flash('Background set successfully!', 'success')
                    except Exception as e:
                        flash(f'Failed to save background: {str(e)}', 'error')
        
        elif action == 'clear_background':
            try:
                db_manager.execute_query("UPDATE pdf_settings SET background_path = NULL WHERE setting_name = 'prescription_background'")
                flash('Background cleared!', 'success')
            except Exception as e:
                flash(f'Failed to clear background: {str(e)}', 'error')
    
    background_path = None
    try:
        result = db_manager.execute_query("SELECT background_path FROM pdf_settings WHERE setting_name = 'prescription_background' LIMIT 1")
        if result and result[0][0]:
            background_path = result[0][0]
    except:
        pass
    
    return render_template('settings.html', background_path=background_path)

# ============ PATIENT MANAGEMENT ROUTES ============

@app.route('/patients')
def patients():
    if 'user' not in session:
        return redirect(url_for('login'))
    
    # Get current month for "New This Month" calculation
    current_month = datetime.now().strftime('%Y-%m')
    
    search = request.args.get('search', '').strip()
    
    if search:
        query = """
            SELECT id, mr_number, name, cnic, mobile, age, gender, created_date 
            FROM patients 
            WHERE name LIKE ? OR mr_number LIKE ? OR cnic LIKE ? OR mobile LIKE ?
            ORDER BY created_date DESC
        """
        search_param = f"%{search}%"
        patients = db_manager.execute_query(query, (search_param, search_param, search_param, search_param))
    else:
        patients = db_manager.execute_query("""
            SELECT id, mr_number, name, cnic, mobile, age, gender, created_date 
            FROM patients 
            ORDER BY created_date DESC
        """)
    
    return render_template('patients.html', patients=patients, search=search, current_month=current_month)

@app.route('/patients/new', methods=['GET', 'POST'])
def new_patient():
    if 'user' not in session:
        return redirect(url_for('login'))
    
    if request.method == 'POST':
        name = request.form.get('name', '').strip()
        cnic = request.form.get('cnic', '').strip()
        mobile = request.form.get('mobile', '').strip()
        dob = request.form.get('dob', '').strip()
        gender = request.form.get('gender', 'Male')
        address = request.form.get('address', '').strip()
        
        if not all([name, cnic, mobile, dob, gender]):
            flash('Please fill all required fields', 'error')
            return render_template('patient_form.html', patient=None)
        
        try:
            int(dob)
            if len(dob) != 4:
                raise ValueError
        except ValueError:
            flash('Please enter valid 4-digit birth year', 'error')
            return render_template('patient_form.html', patient=None)
        
        mr_number = generate_unique_id("MR")
        age = calculate_age(dob)
        
        try:
            patient_id = db_manager.execute_insert('''
                INSERT INTO patients (mr_number, name, cnic, mobile, dob, age, gender, address, created_date)
                VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?)
            ''', (mr_number, name, cnic, mobile, dob, age, gender, address, 
                  datetime.now().strftime('%Y-%m-%d %H:%M:%S')))
            
            flash(f'Patient registered successfully! MR Number: {mr_number}', 'success')
            return redirect(url_for('patients'))
            
        except Exception as e:
            flash(f'Failed to save patient: {str(e)}', 'error')
            return render_template('patient_form.html', patient=None)
    
    return render_template('patient_form.html', patient=None)

@app.route('/patients/<int:patient_id>/edit', methods=['GET', 'POST'])
def edit_patient(patient_id):
    if 'user' not in session:
        return redirect(url_for('login'))
    
    patient_data = db_manager.execute_query("SELECT * FROM patients WHERE id = ?", (patient_id,))
    if not patient_data:  # ✅ FIXED: was patient_ before
        flash('Patient not found', 'error')
        return redirect(url_for('patients'))
    
    patient = patient_data[0]
    
    if request.method == 'POST':
        name = request.form.get('name', '').strip()
        cnic = request.form.get('cnic', '').strip()
        mobile = request.form.get('mobile', '').strip()
        dob = request.form.get('dob', '').strip()
        gender = request.form.get('gender', 'Male')
        address = request.form.get('address', '').strip()
        
        if not all([name, cnic, mobile, dob, gender]):
            flash('Please fill all required fields', 'error')
            return render_template('patient_form.html', patient=patient)
        
        try:
            int(dob)
            if len(dob) != 4:
                raise ValueError
        except ValueError:
            flash('Please enter valid 4-digit birth year', 'error')
            return render_template('patient_form.html', patient=patient)
        
        age = calculate_age(dob)
        
        try:
            db_manager.execute_query('''
                UPDATE patients 
                SET name = ?, cnic = ?, mobile = ?, dob = ?, age = ?, gender = ?, address = ?
                WHERE id = ?
            ''', (name, cnic, mobile, dob, age, gender, address, patient_id))
            
            flash('Patient updated successfully!', 'success')
            return redirect(url_for('patients'))
            
        except Exception as e:
            flash(f'Failed to update patient: {str(e)}', 'error')
            return render_template('patient_form.html', patient=patient)
    
    return render_template('patient_form.html', patient=patient)

@app.route('/patients/<int:patient_id>/delete', methods=['POST'])
def delete_patient(patient_id):
    if 'user' not in session:
        return redirect(url_for('login'))
    
    try:
        db_manager.execute_query("DELETE FROM patients WHERE id = ?", (patient_id,))
        flash('Patient deleted successfully!', 'success')
    except Exception as e:
        flash(f'Failed to delete patient: {str(e)}', 'error')
    
    return redirect(url_for('patients'))

@app.route('/patients/<int:patient_id>/token')
def generate_token(patient_id):
    if 'user' not in session:
        return redirect(url_for('login'))
    
    try:
        patient_data = db_manager.execute_query(
            "SELECT mr_number, name, cnic, mobile, age, gender FROM patients WHERE id = ?", 
            (patient_id,)
        )
        if not patient_data:  # ✅ FIXED: was patient_ before
            flash('Patient not found', 'error')
            return redirect(url_for('patients'))
        
        patient = patient_data[0]
        patient_dict = {
            'mr_number': patient[0],
            'name': patient[1],
            'cnic': patient[2],
            'mobile': patient[3],
            'age': patient[4],
            'gender': patient[5]
        }
        
        from utils.helpers import get_next_daily_token
        token_number = get_next_daily_token(db_manager)
        
        pdf_generator = PDFGenerator(db_manager)
        pdf_bytes = pdf_generator.generate_token_pdf(patient_dict, token_number)
        
        # Show in browser for printing (don't save to file)
        return send_file(
            BytesIO(pdf_bytes),
            mimetype='application/pdf',
            as_attachment=False,  # Show in browser
            download_name=f'token_{token_number}.pdf'
        )
        
    except Exception as e:
        flash(f'Failed to generate token: {str(e)}', 'error')
        return redirect(url_for('patients'))

@app.route('/patients/<int:patient_id>/prescription')
def generate_prescription(patient_id):
    if 'user' not in session:
        return redirect(url_for('login'))
    
    try:
        patient_data = db_manager.execute_query(
            "SELECT mr_number, name, cnic, mobile, age, gender FROM patients WHERE id = ?", 
            (patient_id,)
        )
        if not patient_data:  # ✅ FIXED: was patient_ before
            flash('Patient not found', 'error')
            return redirect(url_for('patients'))
        
        patient = patient_data[0]
        patient_dict = {
            'mr_number': patient[0],
            'name': patient[1],
            'cnic': patient[2],
            'mobile': patient[3],
            'age': patient[4],
            'gender': patient[5]
        }
        
        pdf_generator = PDFGenerator(db_manager)
        pdf_bytes = pdf_generator.generate_prescription_pdf(patient_dict)
        
        # Show in browser for printing (don't save to file)
        return send_file(
            BytesIO(pdf_bytes),
            mimetype='application/pdf',
            as_attachment=False,  # Show in browser
            download_name=f'prescription_{patient_dict["mr_number"]}.pdf'
        )
        
    except Exception as e:
        flash(f'Failed to generate prescription: {str(e)}', 'error')
        return redirect(url_for('patients'))

# ============ DOCTOR MANAGEMENT ROUTES ============

@app.route('/doctors')
def doctors():
    if 'user' not in session:
        return redirect(url_for('login'))
    
    doctors = db_manager.execute_query("""
        SELECT doctor_id, name, specialization, mobile, email, fee 
        FROM doctors 
        ORDER BY name
    """)
    
    return render_template('doctors.html', doctors=doctors)

@app.route('/doctors/add', methods=['POST'])
def add_doctor():
    if 'user' not in session:
        return redirect(url_for('login'))
    
    doctor_id = request.form.get('doctor_id', '').strip() or generate_unique_id("DOC")
    name = request.form.get('name', '').strip()
    specialization = request.form.get('specialization', '').strip()
    mobile = request.form.get('mobile', '').strip()
    email = request.form.get('email', '').strip()
    qualification = request.form.get('qualification', '').strip()
    experience = request.form.get('experience', '0').strip() or "0"
    fee = request.form.get('fee', '0').strip() or "0"
    
    if not name:
        flash('Please enter doctor name', 'error')
        return redirect(url_for('doctors'))
    
    try:
        db_manager.execute_insert('''
            INSERT INTO doctors (doctor_id, name, specialization, mobile, email, 
                               qualification, experience, fee, created_date)
            VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?)
        ''', (doctor_id, name, specialization, mobile, email, qualification, 
              int(experience), float(fee), datetime.now().strftime('%Y-%m-%d %H:%M:%S')))
        
        flash('Doctor added successfully!', 'success')
        
    except Exception as e:
        flash(f'Failed to add doctor: {str(e)}', 'error')
    
    return redirect(url_for('doctors'))

@app.route('/doctors/update', methods=['POST'])
def update_doctor():
    if 'user' not in session:
        return redirect(url_for('login'))
    
    doctor_id = request.form.get('doctor_id', '').strip()
    name = request.form.get('name', '').strip()
    specialization = request.form.get('specialization', '').strip()
    mobile = request.form.get('mobile', '').strip()
    email = request.form.get('email', '').strip()
    qualification = request.form.get('qualification', '').strip()
    experience = request.form.get('experience', '0').strip() or "0"
    fee = request.form.get('fee', '0').strip() or "0"
    
    if not doctor_id:
        flash('No doctor selected', 'error')
        return redirect(url_for('doctors'))
    
    if not name:
        flash('Please enter doctor name', 'error')
        return redirect(url_for('doctors'))
    
    try:
        db_manager.execute_query('''
            UPDATE doctors SET name=?, specialization=?, mobile=?, email=?, 
                             qualification=?, experience=?, fee=?
            WHERE doctor_id=?
        ''', (name, specialization, mobile, email, qualification, 
              int(experience), float(fee), doctor_id))
        
        flash('Doctor updated successfully!', 'success')
        
    except Exception as e:
        flash(f'Failed to update doctor: {str(e)}', 'error')
    
    return redirect(url_for('doctors'))

@app.route('/doctors/delete', methods=['POST'])
def delete_doctor():
    if 'user' not in session:
        return redirect(url_for('login'))
    
    doctor_id = request.form.get('doctor_id', '').strip()
    
    if not doctor_id:
        flash('No doctor selected', 'error')
        return redirect(url_for('doctors'))
    
    try:
        db_manager.execute_query("DELETE FROM doctors WHERE doctor_id=?", (doctor_id,))
        flash('Doctor deleted successfully!', 'success')
    except Exception as e:
        flash(f'Failed to delete doctor: {str(e)}', 'error')
    
    return redirect(url_for('doctors'))

# ============ APPOINTMENT MANAGEMENT ROUTES ============

@app.route('/appointments')
def appointments():
    if 'user' not in session:
        return redirect(url_for('login'))
    
    patients = db_manager.execute_query("SELECT id, name FROM patients ORDER BY name")
    doctors = db_manager.execute_query("SELECT id, name FROM doctors ORDER BY name")
    
    appointments = db_manager.execute_query('''
        SELECT a.appointment_id, p.name, d.name, a.appointment_date, a.appointment_time, a.status
        FROM appointments a
        JOIN patients p ON a.patient_id = p.id
        JOIN doctors d ON a.doctor_id = d.id
        ORDER BY a.appointment_date DESC, a.appointment_time DESC
    ''')
    
    return render_template('appointments.html', patients=patients, doctors=doctors, appointments=appointments)

@app.route('/appointments/schedule', methods=['POST'])
def schedule_appointment():
    if 'user' not in session:
        return redirect(url_for('login'))
    
    appointment_id = request.form.get('appointment_id', '').strip() or generate_unique_id("APT")
    patient_id = request.form.get('patient_id', '').strip()
    doctor_id = request.form.get('doctor_id', '').strip()
    date = request.form.get('date', '').strip()
    time = request.form.get('time', '').strip()
    status = request.form.get('status', 'Scheduled')
    notes = request.form.get('notes', '').strip()
    
    if not all([patient_id, doctor_id, date, time]):
        flash('Please fill all required fields', 'error')
        return redirect(url_for('appointments'))
    
    try:
        db_manager.execute_insert('''
            INSERT INTO appointments (appointment_id, patient_id, doctor_id, appointment_date, 
                                    appointment_time, status, notes, created_date)
            VALUES (?, ?, ?, ?, ?, ?, ?, ?)
        ''', (appointment_id, int(patient_id), int(doctor_id), date, time, status, notes, 
              datetime.now().strftime('%Y-%m-%d %H:%M:%S')))
        
        flash('Appointment scheduled successfully!', 'success')
        
    except Exception as e:
        flash(f'Failed to schedule appointment: {str(e)}', 'error')
    
    return redirect(url_for('appointments'))

@app.route('/appointments/update', methods=['POST'])
def update_appointment():
    if 'user' not in session:
        return redirect(url_for('login'))
    
    appointment_id = request.form.get('appointment_id', '').strip()
    patient_id = request.form.get('patient_id', '').strip()
    doctor_id = request.form.get('doctor_id', '').strip()
    date = request.form.get('date', '').strip()
    time = request.form.get('time', '').strip()
    status = request.form.get('status', 'Scheduled')
    notes = request.form.get('notes', '').strip()
    
    if not appointment_id:
        flash('No appointment selected', 'error')
        return redirect(url_for('appointments'))
    
    if not all([patient_id, doctor_id, date, time]):
        flash('Please fill all required fields', 'error')
        return redirect(url_for('appointments'))
    
    try:
        db_manager.execute_query('''
            UPDATE appointments 
            SET patient_id=?, doctor_id=?, appointment_date=?, appointment_time=?, 
                status=?, notes=?
            WHERE appointment_id=?
        ''', (int(patient_id), int(doctor_id), date, time, status, notes, appointment_id))
        
        flash('Appointment updated successfully!', 'success')
        
    except Exception as e:
        flash(f'Failed to update appointment: {str(e)}', 'error')
    
    return redirect(url_for('appointments'))

@app.route('/appointments/cancel', methods=['POST'])
def cancel_appointment():
    if 'user' not in session:
        return redirect(url_for('login'))
    
    appointment_id = request.form.get('appointment_id', '').strip()
    
    if not appointment_id:
        flash('No appointment selected', 'error')
        return redirect(url_for('appointments'))
    
    try:
        db_manager.execute_query("UPDATE appointments SET status='Cancelled' WHERE appointment_id=?", 
                                (appointment_id,))
        flash('Appointment cancelled successfully!', 'success')
    except Exception as e:
        flash(f'Failed to cancel appointment: {str(e)}', 'error')
    
    return redirect(url_for('appointments'))

# ============ MEDICINE MANAGEMENT ROUTES ============

@app.route('/medicines')
def medicines():
    if 'user' not in session:
        return redirect(url_for('login'))
    
    medicines = db_manager.execute_query("""
        SELECT medicine_id, name, category, stock_quantity, unit_price, expiry_date 
        FROM medicines 
        ORDER BY name
    """)
    
    return render_template('medicines.html', medicines=medicines, now=datetime.now, timedelta=timedelta)

@app.route('/medicines/add', methods=['POST'])
def add_medicine():
    if 'user' not in session:
        return redirect(url_for('login'))
    
    medicine_id = request.form.get('medicine_id', '').strip() or generate_unique_id("MED")
    name = request.form.get('name', '').strip()
    generic = request.form.get('generic_name', '').strip()
    manufacturer = request.form.get('manufacturer', '').strip()
    category = request.form.get('category', '').strip()
    price = request.form.get('unit_price', '0').strip() or "0"
    stock = request.form.get('stock_quantity', '0').strip() or "0"
    expiry = request.form.get('expiry_date', '').strip()
    
    if not name:
        flash('Please enter medicine name', 'error')
        return redirect(url_for('medicines'))
    
    try:
        db_manager.execute_insert('''
            INSERT INTO medicines (medicine_id, name, generic_name, manufacturer, 
                                 category, unit_price, stock_quantity, expiry_date, created_date)
            VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?)
        ''', (medicine_id, name, generic, manufacturer, category, 
              float(price), int(stock), expiry, datetime.now().strftime('%Y-%m-%d %H:%M:%S')))
        
        flash('Medicine added successfully!', 'success')
        
    except Exception as e:
        flash(f'Failed to add medicine: {str(e)}', 'error')
    
    return redirect(url_for('medicines'))

@app.route('/medicines/update', methods=['POST'])
def update_medicine():
    if 'user' not in session:
        return redirect(url_for('login'))
    
    medicine_id = request.form.get('medicine_id', '').strip()
    name = request.form.get('name', '').strip()
    generic = request.form.get('generic_name', '').strip()
    manufacturer = request.form.get('manufacturer', '').strip()
    category = request.form.get('category', '').strip()
    price = request.form.get('unit_price', '0').strip() or "0"
    stock = request.form.get('stock_quantity', '0').strip() or "0"
    expiry = request.form.get('expiry_date', '').strip()
    
    if not medicine_id:
        flash('No medicine selected', 'error')
        return redirect(url_for('medicines'))
    
    if not name:
        flash('Please enter medicine name', 'error')
        return redirect(url_for('medicines'))
    
    try:
        db_manager.execute_query('''
            UPDATE medicines 
            SET name=?, generic_name=?, manufacturer=?, category=?, 
                unit_price=?, stock_quantity=?, expiry_date=?
            WHERE medicine_id=?
        ''', (name, generic, manufacturer, category, float(price), 
              int(stock), expiry, medicine_id))
        
        flash('Medicine updated successfully!', 'success')
        
    except Exception as e:
        flash(f'Failed to update medicine: {str(e)}', 'error')
    
    return redirect(url_for('medicines'))

@app.route('/medicines/delete', methods=['POST'])
def delete_medicine():
    if 'user' not in session:
        return redirect(url_for('login'))
    
    medicine_id = request.form.get('medicine_id', '').strip()
    
    if not medicine_id:
        flash('No medicine selected', 'error')
        return redirect(url_for('medicines'))
    
    try:
        db_manager.execute_query("DELETE FROM medicines WHERE medicine_id=?", (medicine_id,))
        flash('Medicine deleted successfully!', 'success')
    except Exception as e:
        flash(f'Failed to delete medicine: {str(e)}', 'error')
    
    return redirect(url_for('medicines'))

# ============ BILLING MANAGEMENT ROUTES ============

@app.route('/billing')
def billing():
    if 'user' not in session:
        return redirect(url_for('login'))
    
    patients = db_manager.execute_query("SELECT id, name, mr_number FROM patients ORDER BY name")
    doctors = db_manager.execute_query("SELECT id, name, fee FROM doctors ORDER BY name")
    medicines = db_manager.execute_query("SELECT id, name, unit_price, stock_quantity FROM medicines ORDER BY name")
    
    bills = db_manager.execute_query('''
        SELECT b.bill_id, p.name, d.name, b.total_amount, b.payment_status, b.bill_date
        FROM billing b
        JOIN patients p ON b.patient_id = p.id
        JOIN doctors d ON b.doctor_id = d.id
        ORDER BY b.bill_date DESC
    ''')
    
    return render_template('billing.html', patients=patients, doctors=doctors, medicines=medicines, bills=bills)

@app.route('/billing/add_medicine', methods=['POST'])
def add_medicine_to_cart():
    if 'user' not in session:
        return redirect(url_for('login'))
    
    medicine_id = request.form.get('medicine_id', '').strip()
    quantity = request.form.get('quantity', '1').strip()
    
    if not medicine_id:
        return jsonify({'error': 'Please select a medicine'}), 400
    
    try:
        quantity = int(quantity)
        if quantity <= 0:
            return jsonify({'error': 'Please enter valid quantity'}), 400
    except ValueError:
        return jsonify({'error': 'Please enter valid quantity'}), 400
    
    medicine = db_manager.execute_query("SELECT id, name, unit_price, stock_quantity FROM medicines WHERE id = ?", (medicine_id,))
    if not medicine:
        return jsonify({'error': 'Medicine not found'}), 400
    
    medicine = medicine[0]
    if quantity > medicine[3]:
        return jsonify({'error': f'Insufficient stock. Available: {medicine[3]}'}), 400
    
    cart = session.get('cart', [])
    
    existing_item = None
    for item in cart:
        if item['medicine_id'] == medicine_id:
            existing_item = item
            break
    
    if existing_item:
        existing_item['quantity'] += quantity
        existing_item['total'] = existing_item['quantity'] * existing_item['unit_price']
    else:
        cart_item = {
            'medicine_id': medicine[0],
            'medicine_name': medicine[1],
            'quantity': quantity,
            'unit_price': medicine[2],
            'total': quantity * medicine[2]
        }
        cart.append(cart_item)
    
    session['cart'] = cart
    return jsonify({'success': True})

@app.route('/billing/remove_medicine', methods=['POST'])
def remove_medicine_from_cart():
    if 'user' not in session:
        return redirect(url_for('login'))
    
    medicine_id = request.form.get('medicine_id', '').strip()
    
    if not medicine_id:
        return jsonify({'error': 'No medicine selected'}), 400
    
    cart = session.get('cart', [])
    cart = [item for item in cart if item['medicine_id'] != medicine_id]
    session['cart'] = cart
    return jsonify({'success': True})

@app.route('/billing/clear_cart', methods=['POST'])
def clear_cart():
    if 'user' not in session:
        return redirect(url_for('login'))
    
    session['cart'] = []
    return jsonify({'success': True})

@app.route('/billing/calculate', methods=['POST'])
def calculate_billing():
    if 'user' not in session:
        return redirect(url_for('login'))
    
    cart = session.get('cart', [])
    consultation_fee = float(request.form.get('consultation_fee', '0') or '0')
    discount_percent = float(request.form.get('discount', '0') or '0')
    tax_percent = float(request.form.get('tax', '0') or '0')
    
    medicine_subtotal = sum(item['total'] for item in cart)
    subtotal = consultation_fee + medicine_subtotal
    discount_amount = subtotal * (discount_percent / 100)
    taxable_amount = subtotal - discount_amount
    tax_amount = taxable_amount * (tax_percent / 100)
    total_amount = taxable_amount + tax_amount
    
    return jsonify({
        'subtotal': subtotal,
        'discount_amount': discount_amount,
        'tax_amount': tax_amount,
        'total_amount': total_amount
    })

@app.route('/billing/generate', methods=['POST'])
def generate_bill():
    if 'user' not in session:
        return redirect(url_for('login'))
    
    patient_id = request.form.get('patient_id', '').strip()
    doctor_id = request.form.get('doctor_id', '').strip()
    consultation_fee = float(request.form.get('consultation_fee', '0') or '0')
    discount_percent = float(request.form.get('discount', '0') or '0')
    tax_percent = float(request.form.get('tax', '0') or '0')
    payment_method = request.form.get('payment_method', 'Cash')
    cart = session.get('cart', [])
    
    if not patient_id or not doctor_id:
        flash('Please select both patient and doctor', 'error')
        return redirect(url_for('billing'))
    
    if not cart:
        flash('Please add medicines to cart', 'error')
        return redirect(url_for('billing'))
    
    try:
        medicine_cost = sum(item['total'] for item in cart)
        subtotal = consultation_fee + medicine_cost
        discount_amount = subtotal * (discount_percent / 100)
        taxable_amount = subtotal - discount_amount
        tax_amount = taxable_amount * (tax_percent / 100)
        total_amount = taxable_amount + tax_amount
        
        bill_id = generate_unique_id("BILL")
        
        bill_db_id = db_manager.execute_insert('''
            INSERT INTO billing (bill_id, patient_id, doctor_id, consultation_fee, medicine_cost,
                               total_amount, discount, tax, net_amount, payment_method,
                               payment_status, bill_date, created_date)
            VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)
        ''', (bill_id, int(patient_id), int(doctor_id), consultation_fee, medicine_cost,
              subtotal, discount_amount, tax_amount, total_amount, payment_method,
              'Pending', datetime.now().strftime('%Y-%m-%d'),
              datetime.now().strftime('%Y-%m-%d %H:%M:%S')))
        
        for item in cart:
            db_manager.execute_insert('''
                INSERT INTO bill_items (bill_id, medicine_id, quantity, unit_price, total_price, created_date)
                VALUES (?, ?, ?, ?, ?, ?)
            ''', (bill_db_id, item['medicine_id'], item['quantity'], 
                  item['unit_price'], item['total'], datetime.now().strftime('%Y-%m-%d %H:%M:%S')))
        
        for item in cart:
            db_manager.execute_query('''
                UPDATE medicines SET stock_quantity = stock_quantity - ? WHERE id = ?
            ''', (item['quantity'], item['medicine_id']))
        
        session['cart'] = []
        
        flash(f'Bill generated successfully! Bill ID: {bill_id}', 'success')
        
    except Exception as e:
        flash(f'Failed to generate bill: {str(e)}', 'error')
    
    return redirect(url_for('billing'))

@app.route('/billing/search', methods=['POST'])
def search_bills():
    if 'user' not in session:
        return redirect(url_for('login'))
    
    search_term = request.form.get('search_term', '').strip()
    
    if not search_term:
        bills = db_manager.execute_query('''
            SELECT b.bill_id, p.name, d.name, b.total_amount, b.payment_status, b.bill_date
            FROM billing b
            JOIN patients p ON b.patient_id = p.id
            JOIN doctors d ON b.doctor_id = d.id
            ORDER BY b.bill_date DESC
        ''')
    else:
        bills = db_manager.execute_query('''
            SELECT b.bill_id, p.name, d.name, b.total_amount, b.payment_status, b.bill_date
            FROM billing b
            JOIN patients p ON b.patient_id = p.id
            JOIN doctors d ON b.doctor_id = d.id
            WHERE p.name LIKE ? OR b.bill_id LIKE ?
            ORDER BY b.bill_date DESC
        ''', (f"%{search_term}%", f"%{search_term}%"))
    
    bill_list = []
    for bill in bills:
        bill_list.append({
            'bill_id': bill[0],
            'patient_name': bill[1],
            'doctor_name': bill[2],
            'total_amount': float(bill[3]),
            'payment_status': bill[4],
            'bill_date': bill[5]
        })
    
    return jsonify({'bills': bill_list})

@app.route('/billing/delete', methods=['POST'])
def delete_bill():
    if 'user' not in session:
        return redirect(url_for('login'))
    
    bill_id = request.form.get('bill_id', '').strip()
    
    if not bill_id:
        return jsonify({'error': 'No bill selected'}), 400
    
    try:
        db_manager.execute_query("DELETE FROM billing WHERE bill_id = ?", (bill_id,))
        return jsonify({'success': True})
    except Exception as e:
        return jsonify({'error': f'Failed to delete bill: {str(e)}'}), 500

# ============ PRESCRIPTION MANAGEMENT ROUTES ============

@app.route('/prescriptions')
def prescriptions():
    if 'user' not in session:
        return redirect(url_for('login'))
    
    return render_template('prescriptions.html')

@app.route('/prescriptions/search', methods=['POST'])
def search_patients():
    if 'user' not in session:
        return redirect(url_for('login'))
    
    search_term = request.form.get('search_term', '').strip()
    
    if not search_term:
        return jsonify({'error': 'Please enter search term'}), 400
    
    query = """
        SELECT id, mr_number, name, cnic, mobile, age, gender FROM patients 
        WHERE name LIKE ? OR mr_number LIKE ? OR cnic LIKE ? OR mobile LIKE ?
    """
    search_param = f"%{search_term}%"
    
    patients = db_manager.execute_query(query, (search_param, search_param, search_param, search_param))
    
    patient_list = []
    for patient in patients:
        patient_list.append({
            'id': patient[0],
            'mr_number': patient[1],
            'name': patient[2],
            'cnic': patient[3],
            'mobile': patient[4],
            'age': patient[5],
            'gender': patient[6]
        })
    
    return jsonify({'patients': patient_list})

@app.route('/prescriptions/save', methods=['POST'])
def save_prescription():
    if 'user' not in session:
        return redirect(url_for('login'))
    
    patient_id = request.form.get('patient_id', '').strip()
    symptoms = request.form.get('symptoms', '').strip()
    diagnosis = request.form.get('diagnosis', '').strip()
    medicine = request.form.get('medicine', '').strip()
    dosage = request.form.get('dosage', '').strip()
    duration = request.form.get('duration', '').strip()
    advice = request.form.get('advice', '').strip()
    
    if not patient_id:
        return jsonify({'error': 'Please select a patient'}), 400
    
    try:
        prescription_id = generate_unique_id("PRX")
        
        db_manager.execute_insert('''
            INSERT INTO prescriptions (prescription_id, patient_id, prescription_date, 
                                     symptoms, diagnosis, medicine, dosage, duration, advice, created_date)
            VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?)
        ''', (prescription_id, int(patient_id), 
              datetime.now().strftime('%Y-%m-%d'), symptoms, diagnosis, medicine, 
              dosage, duration, advice, datetime.now().strftime('%Y-%m-%d %H:%M:%S')))
        
        return jsonify({'success': True, 'message': f'Prescription saved successfully! ID: {prescription_id}'})
        
    except Exception as e:
        return jsonify({'error': f'Failed to save prescription: {str(e)}'}), 500

@app.route('/prescriptions/print', methods=['POST'])
def print_prescription():
    if 'user' not in session:
        return redirect(url_for('login'))
    
    patient_id = request.form.get('patient_id', '').strip()
    
    if not patient_id:
        return jsonify({'error': 'Please select a patient'}), 400
    
    try:
        patient_data = db_manager.execute_query(
            "SELECT mr_number, name, cnic, mobile, age, gender FROM patients WHERE id = ?", 
            (patient_id,)
        )
        if not patient_data:  # ✅ FIXED: was patient_ before
            return jsonify({'error': 'Patient not found'}), 404
        
        patient = patient_data[0]
        patient_dict = {
            'mr_number': patient[0],
            'name': patient[1],
            'cnic': patient[2],
            'mobile': patient[3],
            'age': patient[4],
            'gender': patient[5]
        }
        
        symptoms = request.form.get('symptoms', '').strip()
        diagnosis = request.form.get('diagnosis', '').strip()
        medicine = request.form.get('medicine', '').strip()
        dosage = request.form.get('dosage', '').strip()
        duration = request.form.get('duration', '').strip()
        advice = request.form.get('advice', '').strip()
        
        patient_dict.update({
            'symptoms': symptoms,
            'diagnosis': diagnosis,
            'medicine': medicine,
            'dosage': dosage,
            'duration': duration,
            'advice': advice
        })
        
        pdf_generator = PDFGenerator(db_manager)
        pdf_bytes = pdf_generator.generate_prescription_pdf_with_data(patient_dict)
        
        return send_file(
            BytesIO(pdf_bytes),
            mimetype='application/pdf',
            as_attachment=False,
            download_name=f'prescription_{patient_dict["mr_number"]}.pdf'
        )
        
    except Exception as e:
        return jsonify({'error': f'Failed to generate prescription: {str(e)}'}), 500

# ============ REPORTS & ANALYTICS ROUTES ============

@app.route('/reports')
def reports():
    if 'user' not in session:
        return redirect(url_for('login'))
    
    total_patients = db_manager.execute_query("SELECT COUNT(*) FROM patients")[0][0]
    total_doctors = db_manager.execute_query("SELECT COUNT(*) FROM doctors")[0][0]
    scheduled_appointments = db_manager.execute_query("SELECT COUNT(*) FROM appointments WHERE status = 'Scheduled'")[0][0]
    total_medicines = db_manager.execute_query("SELECT COUNT(*) FROM medicines")[0][0]
    
    return render_template('reports.html', 
                         total_patients=total_patients,
                         total_doctors=total_doctors,
                         scheduled_appointments=scheduled_appointments,
                         total_medicines=total_medicines,
                         report_type='dashboard')

@app.route('/reports/generate', methods=['POST'])
def generate_report():
    if 'user' not in session:
        return redirect(url_for('login'))
    
    report_type = request.form.get('report_type', 'dashboard')
    
    if report_type == 'dashboard':
        total_patients = db_manager.execute_query("SELECT COUNT(*) FROM patients")[0][0]
        total_doctors = db_manager.execute_query("SELECT COUNT(*) FROM doctors")[0][0]
        scheduled_appointments = db_manager.execute_query("SELECT COUNT(*) FROM appointments WHERE status = 'Scheduled'")[0][0]
        total_medicines = db_manager.execute_query("SELECT COUNT(*) FROM medicines")[0][0]
        
        return render_template('reports.html', 
                             total_patients=total_patients,
                             total_doctors=total_doctors,
                             scheduled_appointments=scheduled_appointments,
                             total_medicines=total_medicines,
                             report_type='dashboard')
    
    elif report_type == 'patient_statistics':
        patients = db_manager.execute_query("SELECT mr_number, name, age, gender, mobile, created_date FROM patients ORDER BY created_date DESC")
        return render_template('reports.html', patients=patients, report_type='patient_statistics')
    
    elif report_type == 'doctor_statistics':
        doctors = db_manager.execute_query("SELECT doctor_id, name, specialization, experience, fee, created_date FROM doctors ORDER BY name")
        return render_template('reports.html', doctors=doctors, report_type='doctor_statistics')
    
    elif report_type == 'appointment_statistics':
        appointments = db_manager.execute_query('''
            SELECT a.appointment_id, p.name, d.name, a.appointment_date, a.appointment_time, a.status
            FROM appointments a
            JOIN patients p ON a.patient_id = p.id
            JOIN doctors d ON a.doctor_id = d.id
            ORDER BY a.appointment_date DESC, a.appointment_time DESC
        ''')
        return render_template('reports.html', appointments=appointments, report_type='appointment_statistics')
    
    elif report_type == 'revenue_report':
        bills = db_manager.execute_query('''
            SELECT b.bill_id, p.name, d.name, b.total_amount, b.net_amount, b.payment_status, b.bill_date
            FROM billing b
            JOIN patients p ON b.patient_id = p.id
            JOIN doctors d ON b.doctor_id = d.id
            ORDER BY b.bill_date DESC
        ''')
        
        total_revenue = sum(bill[4] for bill in bills) if bills else 0
        
        return render_template('reports.html', bills=bills, total_revenue=total_revenue, report_type='revenue_report')
    
    elif report_type == 'medicine_inventory':
        from datetime import datetime as dt
        
        medicines = db_manager.execute_query("SELECT medicine_id, name, category, stock_quantity, unit_price, expiry_date FROM medicines ORDER BY name")
        
        low_stock_count = 0
        expired_count = 0
        
        for medicine in medicines:
            if medicine[3] < 10:
                low_stock_count += 1
            if medicine[5]:
                try:
                    expiry_date = dt.strptime(medicine[5], '%Y-%m-%d').date()
                    if expiry_date < dt.now().date():
                        expired_count += 1
                except:
                    pass
        
        return render_template('reports.html', 
                             medicines=medicines, 
                             low_stock_count=low_stock_count,
                             expired_count=expired_count,
                             report_type='medicine_inventory')
    
    else:
        total_patients = db_manager.execute_query("SELECT COUNT(*) FROM patients")[0][0]
        total_doctors = db_manager.execute_query("SELECT COUNT(*) FROM doctors")[0][0]
        scheduled_appointments = db_manager.execute_query("SELECT COUNT(*) FROM appointments WHERE status = 'Scheduled'")[0][0]
        total_medicines = db_manager.execute_query("SELECT COUNT(*) FROM medicines")[0][0]
        
        return render_template('reports.html', 
                             total_patients=total_patients,
                             total_doctors=total_doctors,
                             scheduled_appointments=scheduled_appointments,
                             total_medicines=total_medicines,
                             report_type='dashboard')

# ============ ADMIN USER MANAGEMENT ROUTES ============

@app.route('/admin/users')
def admin_user_management():
    if 'user' not in session or session['user']['role'] != 'Administrator':
        flash('Access denied', 'error')
        return redirect(url_for('dashboard'))
    
    try:
        users = db_manager.execute_query("SELECT id, username, role, created_date FROM users ORDER BY created_date DESC")
        user_list = []
        for user in users:
            user_list.append({
                'id': user[0],
                'username': user[1],
                'role': user[2],
                'created_date': user[3]
            })
    except Exception as e:
        flash(f'Failed to load users: {str(e)}', 'error')
        user_list = []
    
    return render_template('user_management.html', users=user_list)

@app.route('/admin/users/add', methods=['POST'])
def add_user():
    if 'user' not in session or session['user']['role'] != 'Administrator':
        return jsonify({'error': 'Access denied'}), 403
    
    username = request.form.get('username', '').strip()
    password = request.form.get('password', '').strip()
    role = request.form.get('role', 'Receptionist')
    
    if not all([username, password, role]):
        return jsonify({'error': 'Please fill all fields'}), 400
    
    existing = db_manager.execute_query("SELECT COUNT(*) FROM users WHERE username = ?", (username,))
    if existing[0][0] > 0:
        return jsonify({'error': 'Username already exists'}), 400
    
    try:
        hashed_password = hashlib.sha256(password.encode()).hexdigest()  # ✅ Now hashlib is defined
        db_manager.execute_insert('''
            INSERT INTO users (username, password, role, created_date)
            VALUES (?, ?, ?, ?)
        ''', (username, hashed_password, role, datetime.now().strftime('%Y-%m-%d %H:%M:%S')))
        
        return jsonify({'success': True})
        
    except Exception as e:
        return jsonify({'error': f'Failed to add user: {str(e)}'}), 500

@app.route('/admin/users/update', methods=['POST'])
def update_user():
    if 'user' not in session or session['user']['role'] != 'Administrator':
        return jsonify({'error': 'Access denied'}), 403
    
    user_id = request.form.get('user_id', '').strip()
    username = request.form.get('username', '').strip()
    password = request.form.get('password', '').strip()
    role = request.form.get('role', 'Receptionist')
    
    if not all([user_id, username, role]):
        return jsonify({'error': 'Please fill username and role fields'}), 400
    
    try:
        if password:
            hashed_password = hashlib.sha256(password.encode()).hexdigest()  # ✅ Now hashlib is defined
            db_manager.execute_query('''
                UPDATE users 
                SET username = ?, password = ?, role = ?
                WHERE id = ?
            ''', (username, hashed_password, role, user_id))
        else:
            db_manager.execute_query('''
                UPDATE users 
                SET username = ?, role = ?
                WHERE id = ?
            ''', (username, role, user_id))
        
        return jsonify({'success': True})
        
    except Exception as e:
        return jsonify({'error': f'Failed to update user: {str(e)}'}), 500

@app.route('/admin/users/delete', methods=['POST'])
def delete_user():
    if 'user' not in session or session['user']['role'] != 'Administrator':
        return jsonify({'error': 'Access denied'}), 403
    
    user_id = request.form.get('user_id', '').strip()
    if not user_id:
        return jsonify({'error': 'No user selected'}), 400
    
    try:
        db_manager.execute_query("DELETE FROM users WHERE id = ?", (user_id,))
        return jsonify({'success': True})
    except Exception as e:
        return jsonify({'error': f'Failed to delete user: {str(e)}'}), 500

# ============ ADMIN REGISTRATION KEY MANAGEMENT ============

@app.route('/admin/keys')
def admin_keys():
    if 'user' not in session or session['user']['role'] != 'Administrator':
        flash('Access denied', 'error')
        return redirect(url_for('dashboard'))
    
    try:
        keys = db_manager.execute_query("SELECT * FROM temp_registration_keys ORDER BY created_date DESC")
        key_list = []
        for key in keys:
            key_data = {
                'key': key[1],
                'created_date': key[2],
                'expiry_date': key[3],
                'is_used': key[4],
                'pc_identifier': key[5]
            }
            key_list.append(key_data)
    except Exception as e:
        flash(f'Failed to load keys: {str(e)}', 'error')
        key_list = []
    
    return render_template('admin_keys.html', keys=key_list)

@app.route('/admin/keys/generate', methods=['POST'])
def generate_admin_key():
    if 'user' not in session or session['user']['role'] != 'Administrator':
        return jsonify({'error': 'Access denied'}), 403
    
    try:
        days = int(request.form.get('days', 365))
        if days <= 0:
            return jsonify({'error': 'Please enter valid number of days'}), 400
        
        key = db_manager.generate_temp_registration_key(days)
        if key:
            return jsonify({'success': True, 'key': key})
        else:
            return jsonify({'error': 'Failed to generate key'}), 500
            
    except ValueError:
        return jsonify({'error': 'Please enter valid number of days'}), 400
    except Exception as e:
        return jsonify({'error': f'Failed to generate key: {str(e)}'}), 500

@app.route('/admin/keys/delete', methods=['POST'])
def delete_admin_key():
    if 'user' not in session or session['user']['role'] != 'Administrator':
        return jsonify({'error': 'Access denied'}), 403
    
    key = request.form.get('key', '').strip()
    if not key:
        return jsonify({'error': 'No key provided'}), 400
    
    try:
        db_manager.delete_temp_key(key)
        return jsonify({'success': True})
    except Exception as e:
        return jsonify({'error': f'Failed to delete key: {str(e)}'}), 500

# ============ HOSPITAL SETTINGS ROUTES ============

@app.route('/settings/hospital', methods=['GET', 'POST'])
def hospital_settings():
    if 'user' not in session or session['user']['role'] != 'Administrator':
        flash('Access denied', 'error')
        return redirect(url_for('dashboard'))
    
    if request.method == 'POST':
        settings = {
            'hospital_name': request.form.get('hospital_name', '').strip(),
            'hospital_address': request.form.get('hospital_address', '').strip(),
            'hospital_phone': request.form.get('hospital_phone', '').strip(),
            'hospital_timings': request.form.get('hospital_timings', '').strip(),
            'token_prefix': request.form.get('token_prefix', '').strip(),
            'token_suffix': request.form.get('token_suffix', '').strip(),
            'billing_invoice_prefix': request.form.get('billing_invoice_prefix', '').strip(),
            'billing_invoice_suffix': request.form.get('billing_invoice_suffix', '').strip(),
            'billing_footer_text': request.form.get('billing_footer_text', '').strip()
        }
        
        try:
            for setting_name, setting_value in settings.items():
                db_manager.execute_query('''
                    UPDATE hospital_settings 
                    SET setting_value = ?, updated_date = ?
                    WHERE setting_name = ?
                ''', (setting_value, datetime.now().strftime('%Y-%m-%d %H:%M:%S'), setting_name))
            
            flash('Hospital settings updated successfully!', 'success')
        except Exception as e:
            flash(f'Failed to update settings: {str(e)}', 'error')
    
    # Load current settings
    settings = {}
    try:
        result = db_manager.execute_query("SELECT setting_name, setting_value FROM hospital_settings")
        for row in result:
            settings[row[0]] = row[1]
    except Exception as e:
        flash(f'Failed to load settings: {str(e)}', 'error')
    
    return render_template('hospital_settings.html', settings=settings)
    
# ============ EXCEL IMPORT/EXPORT ROUTES ============

@app.route('/medicines/import', methods=['GET', 'POST'])
def import_medicines():
    if 'user' not in session or session['user']['role'] != 'Administrator':
        flash('Access denied', 'error')
        return redirect(url_for('medicines'))
    
    if request.method == 'POST':
        if 'file' not in request.files:
            flash('No file selected', 'error')
            return redirect(url_for('import_medicines'))
        
        file = request.files['file']
        if file.filename == '':
            flash('No file selected', 'error')
            return redirect(url_for('import_medicines'))
        
        if file and file.filename.endswith('.xlsx'):
            try:
                import pandas as pd
                
                # Read Excel file
                df = pd.read_excel(file)
                
                # Required columns
                required_columns = ['name', 'generic_name', 'manufacturer', 'category', 'unit_price', 'stock_quantity', 'expiry_date']
                if not all(col in df.columns for col in required_columns):
                    flash('Invalid file format. Required columns: name, generic_name, manufacturer, category, unit_price, stock_quantity, expiry_date', 'error')
                    return redirect(url_for('import_medicines'))
                
                # Import medicines
                success_count = 0
                error_count = 0
                
                for _, row in df.iterrows():
                    try:
                        medicine_id = generate_unique_id("MED")
                        db_manager.execute_insert('''
                            INSERT INTO medicines (medicine_id, name, generic_name, manufacturer, 
                                                 category, unit_price, stock_quantity, expiry_date, created_date)
                            VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?)
                        ''', (medicine_id, str(row['name']), str(row['generic_name']), str(row['manufacturer']), 
                              str(row['category']), float(row['unit_price']), int(row['stock_quantity']), 
                              str(row['expiry_date']), datetime.now().strftime('%Y-%m-%d %H:%M:%S')))
                        success_count += 1
                    except Exception as e:
                        error_count += 1
                        print(f"Error importing medicine: {e}")
                
                flash(f'Successfully imported {success_count} medicines. {error_count} errors.', 'success')
                return redirect(url_for('medicines'))
                
            except Exception as e:
                flash(f'Failed to import medicines: {str(e)}', 'error')
                return redirect(url_for('import_medicines'))
        else:
            flash('Please upload an Excel file (.xlsx)', 'error')
            return redirect(url_for('import_medicines'))
    
    return render_template('import_medicines.html')

@app.route('/medicines/download_sample')
def download_sample():
    import pandas as pd
    from io import BytesIO
    
    # Create sample data
    sample_data = {
        'name': ['Paracetamol', 'Amoxicillin', 'Aspirin'],
        'generic_name': ['Acetaminophen', 'Amoxicillin', 'Acetylsalicylic Acid'],
        'manufacturer': ['Pharma Co.', 'Medicine Inc.', 'Health Products Ltd.'],
        'category': ['Pain Relief', 'Antibiotic', 'Anti-inflammatory'],
        'unit_price': [10.50, 25.00, 8.75],
        'stock_quantity': [100, 50, 75],
        'expiry_date': ['2025-12-31', '2024-06-30', '2026-03-15']
    }
    
    df = pd.DataFrame(sample_data)
    
    # Create Excel file in memory
    output = BytesIO()
    with pd.ExcelWriter(output, engine='xlsxwriter') as writer:
        df.to_excel(writer, sheet_name='Medicines', index=False)
    
    output.seek(0)
    
    return send_file(
        output,
        mimetype='application/vnd.openxmlformats-officedocument.spreadsheetml.sheet',
        as_attachment=True,
        download_name='medicine_import_sample.xlsx'
    )

@app.route('/patients/export', methods=['GET', 'POST'])
def export_patients():
    if 'user' not in session:
        return redirect(url_for('login'))
    
    if request.method == 'POST':
        export_format = request.form.get('format', 'pdf')
        date_filter = request.form.get('date_filter', 'all')
        start_date = request.form.get('start_date', '')
        end_date = request.form.get('end_date', '')
        
        # Build query based on filters
        if date_filter == 'today':
            query = "SELECT mr_number, name, cnic, mobile, age, gender, created_date FROM patients WHERE DATE(created_date) = DATE('now') ORDER BY created_date DESC"
            params = ()
        elif date_filter == 'weekly':
            query = "SELECT mr_number, name, cnic, mobile, age, gender, created_date FROM patients WHERE created_date >= DATE('now', '-7 days') ORDER BY created_date DESC"
            params = ()
        elif date_filter == 'monthly':
            query = "SELECT mr_number, name, cnic, mobile, age, gender, created_date FROM patients WHERE created_date >= DATE('now', '-1 month') ORDER BY created_date DESC"
            params = ()
        elif date_filter == 'custom' and start_date and end_date:
            query = "SELECT mr_number, name, cnic, mobile, age, gender, created_date FROM patients WHERE DATE(created_date) BETWEEN ? AND ? ORDER BY created_date DESC"
            params = (start_date, end_date)
        else:
            query = "SELECT mr_number, name, cnic, mobile, age, gender, created_date FROM patients ORDER BY created_date DESC"
            params = ()
        
        patients = db_manager.execute_query(query, params)
        
        if export_format == 'excel':
            import pandas as pd
            from io import BytesIO
            
            # Create DataFrame
            df = pd.DataFrame(patients, columns=['MR Number', 'Name', 'CNIC', 'Mobile', 'Age', 'Gender', 'Created Date'])
            
            # Create Excel file in memory
            output = BytesIO()
            with pd.ExcelWriter(output, engine='xlsxwriter') as writer:
                df.to_excel(writer, sheet_name='Patients', index=False)
            
            output.seek(0)
            
            return send_file(
                output,
                mimetype='application/vnd.openxmlformats-officedocument.spreadsheetml.sheet',
                as_attachment=True,
                download_name=f'patients_export_{datetime.now().strftime("%Y%m%d_%H%M%S")}.xlsx'
            )
        
        elif export_format == 'pdf':
            from reportlab.pdfgen import canvas
            from reportlab.lib.pagesizes import letter, landscape
            from reportlab.lib import colors
            from reportlab.platypus import SimpleDocTemplate, Table, TableStyle, Paragraph, Spacer
            from reportlab.lib.styles import getSampleStyleSheet
            from io import BytesIO
            
            # Create PDF in memory
            buffer = BytesIO()
            doc = SimpleDocTemplate(buffer, pagesize=landscape(letter))
            elements = []
            
            # Title
            styles = getSampleStyleSheet()
            title = Paragraph(f"Patient Export - {date_filter.capitalize()} Report", styles['Title'])
            elements.append(title)
            elements.append(Spacer(1, 12))
            
            # Add current date
            date_text = Paragraph(f"Generated on: {datetime.now().strftime('%Y-%m-%d %H:%M:%S')}", styles['Normal'])
            elements.append(date_text)
            elements.append(Spacer(1, 12))
            
            # Create table data
            data = [['MR Number', 'Name', 'CNIC', 'Mobile', 'Age', 'Gender', 'Created Date']]
            for patient in patients:
                data.append(list(patient))
            
            # Create table
            table = Table(data)
            table.setStyle(TableStyle([
                ('BACKGROUND', (0, 0), (-1, 0), colors.grey),
                ('TEXTCOLOR', (0, 0), (-1, 0), colors.whitesmoke),
                ('ALIGN', (0, 0), (-1, -1), 'CENTER'),
                ('FONTNAME', (0, 0), (-1, 0), 'Helvetica-Bold'),
                ('FONTSIZE', (0, 0), (-1, 0), 14),
                ('BOTTOMPADDING', (0, 0), (-1, 0), 12),
                ('BACKGROUND', (0, 1), (-1, -1), colors.beige),
                ('TEXTCOLOR', (0, 1), (-1, -1), colors.black),
                ('ALIGN', (0, 0), (-1, -1), 'CENTER'),
                ('FONTNAME', (0, 1), (-1, -1), 'Helvetica'),
                ('FONTSIZE', (0, 1), (-1, -1), 12),
                ('TOPPADDING', (0, 1), (-1, -1), 6),
                ('BOTTOMPADDING', (0, 1), (-1, -1), 6),
                ('GRID', (0, 0), (-1, -1), 1, colors.black)
            ]))
            
            elements.append(table)
            
            # Build PDF
            doc.build(elements)
            pdf_bytes = buffer.getvalue()
            buffer.close()
            
            return send_file(
                BytesIO(pdf_bytes),
                mimetype='application/pdf',
                as_attachment=True,
                download_name=f'patients_export_{datetime.now().strftime("%Y%m%d_%H%M%S")}.pdf'
            )
    
    return render_template('export_patients.html')

# Update medicines.html to include import button
# Add this to your medicines.html template

if __name__ == '__main__':
    os.makedirs('static/uploads', exist_ok=True)
    os.makedirs('static/pdfs', exist_ok=True)
    
    app.run(host='0.0.0.0', port=5000, debug=True)