# utils/pdf_generator.py - COMPLETELY FIXED VERSION
import os
from reportlab.pdfgen import canvas
from reportlab.lib.pagesizes import letter, A4, inch
from reportlab.lib.units import inch as rl_inch
from reportlab.lib.utils import ImageReader
from datetime import datetime
from io import BytesIO

class PDFGenerator:
    def __init__(self, db_manager):
        self.db_manager = db_manager
        self.hospital_name = "🏥 MEDICAL CENTER HOSPITAL"
        self.prescription_background = None
        self.load_background_settings()
    
    def set_prescription_background(self, background_path):
        """Set prescription background image"""
        self.prescription_background = background_path
        self.save_background_settings()
    
    def load_background_settings(self):
        """Load background settings from database"""
        try:
            result = self.db_manager.execute_query(
                "SELECT background_path FROM pdf_settings WHERE setting_name = 'prescription_background' LIMIT 1"
            )
            if result and result[0][0]:
                bg_path = result[0][0]
                full_path = os.path.join('static/uploads', bg_path) if not bg_path.startswith('static/') else bg_path
                if os.path.exists(full_path):
                    self.prescription_background = full_path
        except Exception as e:
            print(f"Error loading background: {e}")
            pass
    
    def save_background_settings(self):
        """Save background settings to database"""
        try:
            self.db_manager.execute_query('''
                CREATE TABLE IF NOT EXISTS pdf_settings (
                    id INTEGER PRIMARY KEY AUTOINCREMENT,
                    setting_name TEXT UNIQUE,
                    background_path TEXT,
                    created_date TEXT
                )
            ''')
            
            bg_path = None
            if self.prescription_background:
                if 'static/uploads/' in self.prescription_background:
                    bg_path = os.path.basename(self.prescription_background)
                else:
                    bg_path = self.prescription_background
            
            result = self.db_manager.execute_query(
                "SELECT COUNT(*) FROM pdf_settings WHERE setting_name = 'prescription_background'"
            )
            
            current_time = datetime.now().strftime('%Y-%m-%d %H:%M:%S')
            
            if result[0][0] > 0:
                self.db_manager.execute_query('''
                    UPDATE pdf_settings 
                    SET background_path = ?, created_date = ?
                    WHERE setting_name = 'prescription_background'
                ''', (bg_path, current_time))
            else:
                self.db_manager.execute_insert('''
                    INSERT INTO pdf_settings (setting_name, background_path, created_date)
                    VALUES (?, ?, ?)
                ''', ('prescription_background', bg_path, current_time))
        except Exception as e:
            print(f"Error saving background settings: {e}")
            pass
    
    def generate_token_pdf(self, patient_data, token_number, doctor_name=None):
        """Generate token PDF and return bytes for browser display"""
        try:
            buffer = BytesIO()
            receipt_width = 3 * rl_inch
            receipt_height = 5 * rl_inch
            
            c = canvas.Canvas(buffer, pagesize=(receipt_width, receipt_height))
            center_x = receipt_width / 2
            
            # Hospital header
            c.setFont("Helvetica-Bold", 16)
            c.drawCentredString(center_x, receipt_height - 30, "MEDICAL CENTER HOSPITAL")
            c.line(20, receipt_height - 40, receipt_width - 20, receipt_height - 40)
            
            # Patient information
            c.setFont("Helvetica", 10)
            c.drawString(20, receipt_height - 60, f"Patient Name: {patient_data['name']}")
            c.drawString(20, receipt_height - 75, f"MR Number: {patient_data['mr_number']}")
            c.drawString(20, receipt_height - 90, f"Phone: {patient_data['mobile']}")
            
            # Token number (large and bold)
            c.setFont("Helvetica-Bold", 24)
            c.drawCentredString(center_x, receipt_height - 130, f"TOKEN #{token_number}")
            
            # Doctor name (if provided)
            if doctor_name:
                c.setFont("Helvetica-Bold", 12)
                c.drawCentredString(center_x, receipt_height - 160, f"Doctor: {doctor_name}")
            
            # Additional information
            c.setFont("Helvetica", 10)
            c.drawString(20, receipt_height - 180, "Last Fee: $50.00")
            c.drawString(20, receipt_height - 200, f"Date: {datetime.now().strftime('%Y-%m-%d')}")
            c.drawString(20, receipt_height - 210, f"Time: {datetime.now().strftime('%H:%M:%S')}")
            
            # Instructions
            c.setFont("Helvetica", 9)
            c.drawString(20, receipt_height - 230, "📌 Please keep this token with you")
            c.drawString(20, receipt_height - 242, "during your visit.")
            c.drawString(20, receipt_height - 254, "Show at reception counter.")
            
            # Barcode simulation
            c.rect(20, receipt_height - 285, receipt_width - 40, 20)
            c.setFont("Helvetica", 6)
            c.drawCentredString(center_x, receipt_height - 275, f"|||{patient_data['mr_number']}|||")
            
            # Footer
            c.setFont("Helvetica", 8)
            c.drawCentredString(center_x, 25, "Thank you for choosing our services")
            c.drawCentredString(center_x, 15, f"Generated: {datetime.now().strftime('%H:%M:%S')}")
            
            # Save and return bytes
            c.save()
            pdf_bytes = buffer.getvalue()
            buffer.close()
            return pdf_bytes
            
        except Exception as e:
            raise Exception(f"Failed to generate token PDF: {str(e)}")
    
    def generate_prescription_pdf(self, patient_data, background_image=None):
        """Generate prescription PDF and return bytes for browser display"""
        try:
            buffer = BytesIO()
            c = canvas.Canvas(buffer, pagesize=A4)
            width, height = A4
            
            # Add background if exists
            bg_image = background_image if background_image else self.prescription_background
            if bg_image and os.path.exists(bg_image):
                try:
                    c.drawImage(bg_image, 0, 0, width, height, preserveAspectRatio=False)
                except Exception as e:
                    print(f"Background image error: {e}")
                    pass
            
            # Patient information section
            c.setFont("Helvetica", 12)
            
            # First row: Name, CNIC, Mobile
            c.drawString(70, height - 175, f"{patient_data['name']}")
            c.line(70, height - 177, 220, height - 177)
            
            c.drawString(300, height - 175, f"{patient_data['cnic']}")
            c.line(300, height - 177, 420, height - 177)
            
            c.drawString(500, height - 175, f"{patient_data['mobile']}")
            c.line(500, height - 177, width - 50, height - 177)
            
            # Labels for first row
            c.setFont("Helvetica-Bold", 12)
            c.drawString(20, height - 175, "Name:")
            c.drawString(250, height - 175, "CNIC:")
            c.drawString(450, height - 175, "Mobile:")
            
            # Second row: Gender, Age, Weight
            c.setFont("Helvetica", 12)
            c.drawString(70, height - 190, f"{patient_data['gender']}")
            c.line(70, height - 192, 150, height - 192)
            
            c.drawString(200, height - 190, f"{patient_data['age']} years")
            c.line(200, height - 192, 280, height - 192)
            
            c.drawString(350, height - 190, "_________ kg")
            c.line(350, height - 192, width - 50, height - 192)
            
            # Labels for second row
            c.setFont("Helvetica-Bold", 12)
            c.drawString(20, height - 190, "Gender:")
            c.drawString(160, height - 190, "Age:")
            c.drawString(300, height - 190, "Weight:")
            
            # Third row: MR Number and Date
            c.setFont("Helvetica", 12)
            c.drawString(70, height - 160, f"{patient_data['mr_number']}")
            c.line(70, height - 162, 180, height - 162)
            
            c.drawString(500, height - 160, f"{datetime.now().strftime('%Y-%m-%d')}")
            c.line(500, height - 162, width - 50, height - 162)
            
            # Labels for third row
            c.setFont("Helvetica-Bold", 12)
            c.drawString(20, height - 160, "MR#:")
            c.drawString(450, height - 160, "Date:")
            
            # Save and return bytes
            c.save()
            pdf_bytes = buffer.getvalue()
            buffer.close()
            return pdf_bytes
            
        except Exception as e:
            raise Exception(f"Failed to generate prescription PDF: {str(e)}")
    
    def generate_prescription_pdf_with_data(self, patient_data):
        """Generate prescription PDF with patient and prescription data"""
        try:
            buffer = BytesIO()
            c = canvas.Canvas(buffer, pagesize=A4)
            width, height = A4
            
            # Add background if exists
            if self.prescription_background and os.path.exists(self.prescription_background):
                try:
                    c.drawImage(self.prescription_background, 0, 0, width, height, preserveAspectRatio=False)
                except Exception as e:
                    print(f"Background image error: {e}")
                    pass
            
            # Patient information section
            c.setFont("Helvetica", 12)
            
            # First row: Name, CNIC, Mobile
            c.drawString(70, height - 175, f"{patient_data['name']}")
            c.line(70, height - 177, 220, height - 177)
            
            c.drawString(300, height - 175, f"{patient_data['cnic']}")
            c.line(300, height - 177, 420, height - 177)
            
            c.drawString(500, height - 175, f"{patient_data['mobile']}")
            c.line(500, height - 177, width - 50, height - 177)
            
            # Labels for first row
            c.setFont("Helvetica-Bold", 12)
            c.drawString(20, height - 175, "Name:")
            c.drawString(250, height - 175, "CNIC:")
            c.drawString(450, height - 175, "Mobile:")
            
            # Second row: Gender, Age, Weight
            c.setFont("Helvetica", 12)
            c.drawString(70, height - 190, f"{patient_data['gender']}")
            c.line(70, height - 192, 150, height - 192)
            
            c.drawString(200, height - 190, f"{patient_data['age']} years")
            c.line(200, height - 192, 280, height - 192)
            
            c.drawString(350, height - 190, "_________ kg")
            c.line(350, height - 192, width - 50, height - 192)
            
            # Labels for second row
            c.setFont("Helvetica-Bold", 12)
            c.drawString(20, height - 190, "Gender:")
            c.drawString(160, height - 190, "Age:")
            c.drawString(300, height - 190, "Weight:")
            
            # Third row: MR Number and Date
            c.setFont("Helvetica", 12)
            c.drawString(70, height - 160, f"{patient_data['mr_number']}")
            c.line(70, height - 162, 180, height - 162)
            
            c.drawString(500, height - 160, f"{datetime.now().strftime('%Y-%m-%d')}")
            c.line(500, height - 162, width - 50, height - 162)
            
            # Labels for third row
            c.setFont("Helvetica-Bold", 12)
            c.drawString(20, height - 160, "MR#:")
            c.drawString(450, height - 160, "Date:")
            
            # Prescription details
            y_position = height - 220
            
            # Symptoms
            c.setFont("Helvetica-Bold", 14)
            c.drawString(50, y_position, "Symptoms:")
            y_position -= 20
            c.setFont("Helvetica", 12)
            c.drawString(50, y_position, patient_data.get('symptoms', ''))
            y_position -= 30
            
            # Diagnosis
            c.setFont("Helvetica-Bold", 14)
            c.drawString(50, y_position, "Diagnosis:")
            y_position -= 20
            c.setFont("Helvetica", 12)
            c.drawString(50, y_position, patient_data.get('diagnosis', ''))
            y_position -= 30
            
            # Medicine
            c.setFont("Helvetica-Bold", 14)
            c.drawString(50, y_position, "Medicine:")
            y_position -= 20
            c.setFont("Helvetica", 12)
            c.drawString(50, y_position, patient_data.get('medicine', ''))
            y_position -= 30
            
            # Dosage & Duration
            c.setFont("Helvetica-Bold", 14)
            c.drawString(50, y_position, "Dosage & Duration:")
            y_position -= 20
            c.setFont("Helvetica", 12)
            dosage_duration = f"{patient_data.get('dosage', '')} for {patient_data.get('duration', '')}"
            c.drawString(50, y_position, dosage_duration)
            y_position -= 30
            
            # Advice
            c.setFont("Helvetica-Bold", 14)
            c.drawString(50, y_position, "Advice:")
            y_position -= 20
            c.setFont("Helvetica", 12)
            c.drawString(50, y_position, patient_data.get('advice', ''))
            y_position -= 30
            c.drawString(50, y_position, "__________________________________________________")
            y_position -= 30
            
            # Doctor's Signature
            c.setFont("Helvetica-Bold", 14)
            c.drawString(50, y_position, "Doctor's Signature: ________________________")
            
            # Save and return bytes
            c.save()
            pdf_bytes = buffer.getvalue()
            buffer.close()
            return pdf_bytes
            
        except Exception as e:
            raise Exception(f"Failed to generate prescription PDF: {str(e)}")